/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Collections;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Drawing;

using Borland.Eco.Internal.DefaultImpl;
using Borland.Eco.DataRepresentation;
using Borland.Eco.UmlRt;
using Borland.Eco.Globalization;

namespace Borland.Eco.Persistence
{
	[ToolboxBitmap(typeof(PersistenceMapperMultiDb), "Borland.Eco.Persistence.SyncHandler.bmp")]
	[TypeConverter("Borland.Eco.Persistence.SyncHandlerConverter, Borland.Eco.Persistence.Design")]
	public class SyncHandler: AbstractSyncHandler, ISyncSink
	{
		private readonly Queue m_CommitQueue = new Queue();
		private readonly ArrayList m_Blocks = new ArrayList();
		private Guid m_Guid = Guid.NewGuid();
		private int m_FirstVersion = 1;  // First commitblock is version 1
		private int m_CurrentVersion = 0;
		private int m_HistoryLength = 10000;
		private const int HistoryDelta = 100;

		/// <summary>
		/// <para>The number of UpdateDatase operations to save in Synchronization history.</para>
		/// </summary>
		[LocalizableCategory(typeof(PersistenceStringRes), "sCategorySynchronization")]
		[LocalizableDescription(typeof(PersistenceStringRes), "sPropertyHistoryLength")]
		[DefaultValue(100000)]
		public int HistoryLength
		{
			get { return m_HistoryLength; }
			set { m_HistoryLength = value; }
		}

		// Must be called with "this" aready locked
		private void EmptyCommitQueue()
		{
			lock(m_CommitQueue.SyncRoot)
			{
				while (m_CommitQueue.Count > 0)
				{
					m_Blocks.Add(m_CommitQueue.Dequeue());
					m_CurrentVersion++;
				}
			}
		}

		#region ISyncSink implementation

		object ISyncSink.SyncRoot
		{
			get { return m_CommitQueue.SyncRoot; }
		}

		SyncVersion ISyncSink.submit(CommitBlock commitBlock)
		{
			m_CommitQueue.Enqueue(commitBlock);
			return new SyncVersion(m_CurrentVersion + m_CommitQueue.Count, m_Guid);
		}
		#endregion


		private static void AddNonEmbeddedChanges(DBChangeCollection changeCollection, ObjectContents oc, IEcoTypeSystem typeSystem)
		{
			IClass umlClass = typeSystem.AllClasses[oc.ObjectId.ClassId] as IClass;
			for (int m = 0; m < oc.MemberCount; m++)
			{
				IStructuralFeature sf = umlClass.EcoClass.AllStructuralFeatures[m];

				if ((sf.FeatureType == FeatureType.AssociationEnd) &&
					sf.EcoStructuralFeature.Persistent &&
					sf.EcoStructuralFeature.IsStoredInObject)
				{
					if (oc.IsMemberAssigned(m))
					{
						SingleAssociationEnd ae = oc.GetMember(m) as SingleAssociationEnd; // the only association ends that  are storedinobject
						if (ae != null)
						{
							ObjectId idOfOtherEnd = ae.Id;
							if (idOfOtherEnd != null)
							{
								int indexOfOppositeEnd = ((IAssociationEnd)sf).OppositeEnd.EcoAssociationEnd.IndexInAllStructuralFeatures;
								DBChange dbc = changeCollection[idOfOtherEnd];
								dbc.MemberChanged(indexOfOppositeEnd);
							}
						}
					}
				}
			}
		}

		public override SyncVersion CurrentSyncVersion()
		{
			EmptyCommitQueue();
			return new SyncVersion(m_CurrentVersion, m_Guid);
		}

		public override DBChangeCollection GetChangesSince(SyncVersion fromVersion, SyncVersion[] exclude, out SyncVersion lastSyncVersion, IEcoTypeSystem typeSystem)
		{

			ArrayList blockList = new ArrayList();
			DBChangeCollection result = new DBChangeCollection();

			// Check all version from thsi syncserver
			if (fromVersion.Guid != m_Guid)
				result.All = true;
			foreach (SyncVersion s in exclude)
				if (s.Guid != m_Guid)
					result.All = true;

			lock(this)
			{
				EmptyCommitQueue();
				lastSyncVersion = new SyncVersion(m_CurrentVersion, m_Guid);
				if (fromVersion.Version + 1 < m_FirstVersion) // fromVersion is not inclusive, therefore +1
					result.All = true;
				if ((result.All) || (fromVersion.Version >= m_CurrentVersion))
					return result;

				ArrayList blocklist = new ArrayList(m_CurrentVersion - fromVersion.Version);
				for (int i = fromVersion.Version + 1; i <= m_CurrentVersion; i++)
						if ((exclude != null) && !((IList)exclude as IList).Contains(i))  // FIXME other contains
							blockList.Add(m_Blocks[i - m_FirstVersion]);
				if (m_Blocks.Count > (m_HistoryLength + HistoryDelta))
				{
					m_Blocks.RemoveRange(0, HistoryDelta);
					m_FirstVersion += HistoryDelta;
				}
			}

			foreach (CommitBlock cb in blockList)
			{
				foreach (ObjectContents oc in cb.Changes)
				{
					DBChange dbc = result[oc.ObjectId];
					if (oc.ExistenceState == ExistenceState.New)
						dbc.ObjectCreated();
					else if (oc.ExistenceState == ExistenceState.Deleted)
						dbc.ObjectDeleted();
					else
						dbc.EmbeddedMembersChanged = true;
					AddNonEmbeddedChanges(result, oc, typeSystem);
				}

				foreach (ObjectContents oc in cb.OldValues)
					AddNonEmbeddedChanges(result, oc, typeSystem);
			}
			return result;
		}
	}
}
